const remote = require('electron').remote;
const config = require('../config.js');

const Artemis = function() {
	this.init();
};
Artemis.prototype = {
	query: null,
	results: [],
	resultList: null,
	resultIndex: -1,
	prevResultIndex: -1,
	extractors: [],
	init: function() {
		this.loadExtractors();
		this.query = document.getElementById('query');
		this.resultList = document.getElementById('results');
		this.window = remote.getCurrentWindow();
		this.updateWindowSize();
		this.query.focus();

		document.body.addEventListener('keydown', _ => {
			if(_.keyCode === 37) { // arrow  left
				if(this.resultIndex > -1 && this.resultIndex < this.results.length)
					this.results[this.resultIndex].leftAction();
			} else if(_.keyCode === 39) { // arrow  right
				if(this.resultIndex > -1 && this.resultIndex < this.results.length)
					this.results[this.resultIndex].rightAction();
			} 
		})
		document.body.addEventListener('keyup', _ => {
			//console.log(_.keyCode);
			if(_.keyCode === 27) { // ESC
				this.window.hide();
			} else if(_.keyCode === 40) { // arrow down
				this.nextResult();
			} else if(_.keyCode === 38) { // arrow  up
				this.prevResult();
			} else if(_.keyCode === 13) { // enter
				if(_.shiftKey)
					this.results[this.resultIndex].secondaryAction();
				else
					this.results[this.resultIndex].primaryAction();
			}
		});

		this.query.addEventListener('input', _ => {
			this.results.forEach(_ => _.destroy());
			this.results = [];
			this.resultIndex = -1;

			this.extractors.forEach(e => {
				e.extract(this.query.value);
				e.results.forEach(result => this.results.push(result));
				
				Promise.all(e.resultPromises).then(results => {
					this.results = this.results.concat(results);
					this.renderResults(results);
				});
				this.resultList.innerHTML = '';
				this.renderResults(this.results);
			});
		});

		document.addEventListener('nextResult', _ => {
			this.nextResult();
		});
		document.addEventListener('prevResult', _ => {
			this.prevResult();
		});
		document.addEventListener('clearQuery', _ => {
			this.query.value = ''
			this.results.forEach(_ => _.destroy())
			this.results = [];
			this.resultIndex = -1;
			this.resultList.innerHTML = '';
			this.renderResults(this.results);
		})
	},
	loadExtractors: function() {
		config.extractors.forEach(extractor => {
			const Extractor = require('./Extractors/' + extractor + '.js');
			this.extractors.push(new Extractor());
		});
	},
	updateWindowSize: function() {		
		this.window.setContentSize(this.window.getContentSize()[0], Math.min(document.body.clientHeight + 10, 800));
	},
	selectResult: function(n) {
		document.body.removeChild(this.resultList);
		const items = Array.from(this.resultList.getElementsByTagName('li'));
		items.forEach((item, i) => {
			if(i === n) {
				item.classList.add('selected');
				this.results[i].onselect();
				this.resultIndex = n;
			}
			else if(i === this.prevResultIndex) {
				item.classList.remove('selected');
				this.results[i].onunselect();
			}
		});
		this.prevResultIndex = n;
		document.body.appendChild(this.resultList);
		document.body.scrollTop = this.resultList.getElementsByClassName('selected')[0].offsetTop - document.getElementById('mainbar').clientHeight - 200;
	},
	nextResult: function() {
		if(this.resultIndex < this.results.length-1)
			this.resultIndex++;
		else
			this.resultIndex = 0;
		this.selectResult(this.resultIndex);
	},
	prevResult: function() {
		if(this.resultIndex > 0)
			this.resultIndex--;
		else
			this.resultIndex = this.results.length-1;
		this.selectResult(this.resultIndex);
	},
	renderResults: function(results) {
		results.forEach((result, i) => {
			const resultDom = result.getHTML();
			resultDom.addEventListener('click', _ => {
				if(this.resultIndex !== i)
					this.selectResult(i);
			});
			this.resultList.appendChild(resultDom);
		});
		setTimeout(_ => {
			this.updateWindowSize()
			if(this.results.length > 0)
				this.selectResult(0)
		}, 100);
	}
};
module.exports = Artemis;